/***********************************************************************
   (C) Copyright, 1990 by Dean Rubine, Carnegie Mellon University
    Permission to use this code for noncommercial purposes is hereby granted.
    Permission to copy and distribute this code is hereby granted provided
    this copyright notice is retained.  All other rights reserved.
 **********************************************************************/

#include <Types.h>
#ifdef MAC
#include <Memory.h>
#include <SANE.h>
#else
#include <math.h>
#include <memory.h>
#endif
#include <stdio.h>
#include <stdlib.h>

#include "bitvector.h"
#include "matrix.h"
#include "fv.h"	

#ifdef MAC
/* in order to be abe to use the library in code resources (no globals...) */
#define dist_sq_threshold DIST_SQ_THRESHOLD
#define se_th_rolloff SE_TH_ROLLOFF

#else
const FLOAT dist_sq_threshold = DIST_SQ_THRESHOLD;
const FLOAT se_th_rolloff = SE_TH_ROLLOFF;
double invcos2();
static FuncTab acos2;
#endif

#define	EPS	(1.0e-4)


FV
FvAlloc ()
{
	register FV fv;
	fv = (FV) NewPtr (sizeof (struct fv));
	fv->y = NewVector(NFEATURES);
	FvInit(fv);
	return fv;
}

void    
FvFree (FV fv)
{
    FreeVector (fv->y);
#ifdef MAC
	DisposPtr ((Ptr) (void*) fv);
#else
	free ( (void*) fv)
#endif
}

void
FvInit (fv)
register FV fv;
{
	register int i;

	fv->npoints = 0;
	fv->initial_sin = fv->initial_cos = 0.0;
	fv->maxv = 0;
	fv->path_r = 0;
	fv->path_th = 0;
	fv->abs_th = 0;
	fv->sharpness = 0;
	fv->maxv = 0;
	fv->path_z = 0;
	for (i=0; i<5; i++)
		fv->path_f[i] = 0.0;
	for(i = 0; i < NFEATURES; i++)
		fv->y[i] = 0.0;
}

void
FvAddPoint (fv, x, y, z, f, t)
register FV fv; int x, y, z; FingerBend* f; long t;
{
	FLOAT dx1, dy1, magsq1;
	FLOAT th, absth, d;
#ifdef USE_TIME
	FLOAT v;
#endif
	long lasttime;
	int i;

	if (++fv->npoints == 1) {
		fv->starttime = fv->endtime = t;
		fv->startx = fv->endx = x;
		fv->minx = fv->maxx = x;
		fv->starty = fv->endy = y;
		fv->miny = fv->maxy = y;
		fv->endx = x; fv->endy = y;
		
		fv->minz = fv->maxz = z;
		fv->last_z = z; 
		
		for (i=0; i<5; i++) {
			fv->minf[i] = fv->maxf[i] = f[i];
			fv->last_f[i] = f[i];
		}
			
		return;
	}
/* 1 */
	dx1 = x - fv->endx; 
	dy1 = y - fv->endy;
	magsq1 = dx1 * dx1 + dy1 * dy1;

	if ( magsq1 <= dist_sq_threshold
		&& fabs(fv->last_z - z) <= 1
		) { // tester egalement les doigts ***
		fv->npoints--;
		return;		/* ignore this point */
	}

	if (x < fv->minx) fv->minx = x;
	if (x > fv->maxx) fv->maxx = x;
	if (y < fv->miny) fv->miny = y;
	if (y > fv->maxy) fv->maxy = y;

	if (z < fv->minz) fv->minz = z;
	if (z > fv->maxz) fv->maxz = z; 
	fv->path_z += abs(fv->last_z - z);
	fv->last_z = z;
/* 2 */	
	for (i=0; i<5; i++) {
		if (((int) f[i]) < fv->minf[i]) fv->minf[i] = (int) f[i];
		if (((int) f[i]) > fv->maxf[i]) fv->maxf[i] = (int) f[i];
		fv->path_f[i] = fv->path_f[i] + abs(fv->last_f[i] - (int) f[i]);
		fv->last_f[i] = (int) f[i];
	}
	
	lasttime = fv->endtime;
	fv->endtime = t;

	d = sqrt(magsq1);
	fv->path_r += d;
/* 3 */	
	/* calculate initial theta */
	if(fv->npoints == 3 || (fv->npoints > 3 && fv->path_r <= 0.0)) {
		FLOAT magsq, dx, dy, recip;
		dx = x - fv->startx; 
		dy = y - fv->starty;
		magsq = dx * dx + dy * dy;
		if(magsq > dist_sq_threshold)  {
			/* find angle w.r.t. positive x axis e.g. (1,0) */
			recip = 1 / sqrt(magsq);
			fv->initial_cos = dx * recip;
			fv->initial_sin = dy * recip;
		}
	}

	if(fv->npoints >= 3) {
			FLOAT tamp1 = dx1 * fv->dy2 - fv->dx2 * dy1;
			FLOAT tamp2 = dx1 * fv->dx2 + dy1 * fv->dy2;
			th = absth = atan2(tamp1, tamp2);
			if(absth < 0) absth = -absth;
			fv->path_th += th;
			fv->abs_th += absth;
			fv->sharpness += th*th;
#ifdef PF_MAXV
			v = d / (fv->endtime - lasttime);
			if (fv->endtime > lasttime && v > fv->maxv)
				fv->maxv = v;
#endif
	}

	fv->endx = x; fv->endy = y;
	fv->dx2 = dx1; fv->dy2 = dy1;
	fv->magsq2 = magsq1;
}

Vector
FvCalc(fv)
register FV fv;
{
	FLOAT bblen, selen, factor;

	if(fv->npoints <= 1)
		return fv->y;		/* a feature vector of all zeros */

	fv->y[PF_INIT_COS] = fv->initial_cos;
	fv->y[PF_INIT_SIN] = fv->initial_sin;

	bblen = hypot(fv->maxx - fv->minx, fv->maxy - fv->miny);

	fv->y[PF_BB_LEN] = bblen;

	if(bblen * bblen > dist_sq_threshold) 
		fv->y[PF_BB_TH] = atan2(fv->maxy - fv->miny,
					fv->maxx - fv->minx);

	selen = hypot(fv->endx - fv->startx, fv->endy - fv->starty);
	fv->y[PF_SE_LEN] = selen;

	factor = selen * selen / se_th_rolloff;
	if(factor > 1.0) factor = 1.0;
	factor = selen > EPS ? factor/selen : 0.0;
	fv->y[PF_SE_COS] = (fv->endx - fv->startx) * factor;
	fv->y[PF_SE_SIN] = (fv->endy - fv->starty) * factor;

	fv->y[PF_LEN] =  fv->path_r;
	fv->y[PF_TH] = fv->path_th;
	fv->y[PF_ATH] = fv->abs_th;
	fv->y[PF_SQTH] = fv->sharpness;

#ifdef PF_DUR
	fv->y[PF_DUR] = (fv->endtime - fv->starttime)*.01;
#endif

#ifdef PF_MAXV
	fv->y[PF_MAXV] = fv->maxv * 10000;
#endif
	
	fv->y[PF_Z_AMPL] = fv->maxz - fv->minz;
	fv->y[PF_Z_LEN] = fv->path_z;
	
	fv->y[PF_T_AMPL] = fv->maxf[0] - fv->minf[0];
	fv->y[PF_T_LEN] = fv->path_f[0];

	fv->y[PF_I_AMPL] = fv->maxf[1] - fv->minf[1];
	fv->y[PF_I_LEN] = fv->path_f[1];

	fv->y[PF_M_AMPL] = fv->maxf[2] - fv->minf[2];
	fv->y[PF_M_LEN] = fv->path_f[2];

	fv->y[PF_AN_AMPL] = fv->maxf[3] - fv->minf[3];
	fv->y[PF_AN_LEN] = fv->path_f[3];

	fv->y[PF_AU_AMPL] = fv->maxf[4] - fv->minf[4];
	fv->y[PF_AU_LEN] = fv->path_f[4];

	return fv->y;
}

/*
  given m = cos^2(t) * sign(cos(t)) return t
  m must be between -1 and 1
  t will be between 0 and pi 
*/

#ifndef MAC
FLOAT
invcos2(m) FLOAT m;
{
	FLOAT x = m < 0 ? -sqrt(-m) : sqrt(m);
	FLOAT y = sqrt( 1.0 - x * x);	/ * y = sin (acos(x)) * /
	return atan2(y, x);
}
#endif
